#include-once
#cs
#include <GuiListBox.au3>
#include <Array.au3>
#include <WinAPISysWin.au3>
#include <GUIConstantsEx.au3>
#include <WindowsConstants.au3>
#include <WinAPIShellEx.au3>
#include <GuiScrollBars.au3>
#ce

; #INDEX# =======================================================================================================================
; Title .........: AcInput UDF
; AutoIt Version : 3.3.14.5
; Language ......: English
; Author(s) .....: ISI360
; UDF Version ...: 1.3
; Description ...: With this UDF you can add an autocomplete list to an Input Control. You can add an Array of keywoards to the Control
;				   and while your typing the results are displayed in an list under the control. You can select an item in the autocomplete list with
;				   the mouse or ENTER key and change the selection with the arrow up/down keys. The autocomplete list can be hidden with the ESC key.
;
;				   NOTES:
;					- THIS UDF WORKS ONLY CORRECT IN GUI-EVENT MODE !!!!
;					- THIS UDF REQUIRES Lars GUIRegisterMsg20 UDF !!!! You can get it from https://www.autoitscript.com/forum/topic/195151-guiregistermsg20-subclassing-made-easy
;					- This UDF temporary registers a global hotkey via HotKeySet to the ESC key when the autocomplete GUI is shown. It will be removed when its hidden again.
; ===============================================================================================================================

; #CURRENT# =====================================================================================================================
;_AcInputUDF_Initialize
;_AcInputUDF_RegisterControl
;_AcInputUDF_UnRegisterControl
;_AcInputUDF_ControlSetAcList
;_AcInputUDF_IsControlRegistered
;_AcInputUDF_HideAcList
;_AcInputUDF_AutocompleteSelectedItem
;_AcInputUDF_SelectNextItemInList
;_AcInputUDF_SelectPrevItemInList
;_AcInputUDF_ListboxSetColour
;_AcInputUDF_ListboxSetFont
; ===============================================================================================================================

; #INTERNAL_USE_ONLY# ===========================================================================================================
;_AcInputUDF_CheckInput
;_AcInputUDF_AcCheckFocusAdlib
;_AcInputUDF_ReposGui
;_AcInputUDF_ShutdownUDF
;_AcInputUDF_ListBoxUpdateView
; ===============================================================================================================================


; #VARIABLES# ===================================================================================================================
#cs
	$AcInputUDF_RegisteredControlsArray
	[i][0]   - Handle to the registered input control
	[i][1]   - Autocompletelist (Array)
	[i][2]   - Owner GUI Hwnd of the control
	[i][3]   - OnClick Func for the listbox
#ce

Global $AcInputUDF_RegisteredControlsArray[0][4]
Global $AcInputUDF_RegisteredControlsAsString ;Elements from $AcInputUDF_RegisteredControlsArray[i][0] as string, seperated with "|". (For faster compare)
Global $AcInputUDF_AutoListMaxHeight = 200 ;Max height of the autocomplete list. If the list is over this value a scrollbar will be displayed.
Global $AcInputUDF_RegisteredCallBackFunc
Global $AcInputUDF_ACListGUIHandle
Global $AcInputUDF_ACListControlHandle
Global $AcInputUDF_ACListLastInsertedText
Global $AcInputUDF_ACListControlPos
Global $AcInputUDF_LastActivatedControl
Global Const $AcInputUDF_DebugMode = False ;Use True to display the dummy focus control in the guis
; ===============================================================================================================================

; #Example CallBackFunc# ========================================================================================================
;You can use this example for _AcInputUDF_Initialize and adjust it to your needs
#cs

	Func _AcInputUDF_CallbackFunc($hWnd, $iMsg, $iwParam, $ilParam)
	Switch $hWnd

	Case _AcInputUDF_IsControlRegistered($hWnd)

	Switch $iMsg

	Case $WM_KEYDOWN
	;UP KEY
	If $iwParam = 38 Then
	_AcInputUDF_SelectPrevItemInList()
	Return -1 ;Stop here. Prevent from sending the key to the control
	EndIf

	;DOWN KEY
	If $iwParam = 40 Then
	_AcInputUDF_SelectNextItemInList()
	Return -1 ;Stop here. Prevent from sending the key to the control
	EndIf

	;Input Checker for Autocomplete
	_AcInputUDF_CheckInput(_WinAPI_GetDlgCtrlID($hWnd))

	Case $WM_KEYUP
	;ENTER KEY
	If $iwParam = 13 Then _AcInputUDF_AutocompleteSelectedItem()

	EndSwitch

	EndSwitch

	Return $GUI_RUNDEFMSG
	EndFunc   ;==>_AcInputUDF_CallbackFunc

#ce
; ===============================================================================================================================

; #FUNCTION# ====================================================================================================================
; Name...........: _AcInputUDF_Initialize
; Description ...: Initializes the AcInput UDF.
; Syntax.........: _AcInputUDF_Initialize($MainGUI = "", $CallBackFunc = "")
; Parameters ....: $CallBackFunc    - The callback funtion for UDF events.
; Return values .: Success          - 1
;                  Failure          - -1
; Author ........: ISI360
; Modified.......:
; Remarks .......:
; Related .......:
; Link ..........:
; Example .......:
; ===============================================================================================================================
Func _AcInputUDF_Initialize($CallBackFunc = "")
	If $CallBackFunc = "" Then Return -1
	If IsString($CallBackFunc) Then $CallBackFunc = Execute($CallBackFunc) ;The FuncName should be defined without any quotes!
	If IsHWnd($AcInputUDF_ACListGUIHandle) Then Return -1 ;UDF already initiated

	;Create needed controls
	$AcInputUDF_ACListGUIHandle = GUICreate("", 280, 160, 10, 62, $WS_POPUP, BitOR($WS_EX_TOOLWINDOW, $WS_EX_TOPMOST, $WS_EX_LAYERED))
	GUISetBkColor(0xABCDEF, $AcInputUDF_ACListGUIHandle)
	$AcInputUDF_ACListControlHandle = GUICtrlCreateList("", 0, 0, 280, 150, BitOR(0x00100000, 0x00200000))
	_WinAPI_SetLayeredWindowAttributes($AcInputUDF_ACListGUIHandle, 0xABCDEF, 250) ;Make Gui invisible

	$AcInputUDF_ACListGUIHandle_EnterDummy = GUICtrlCreateDummy()
	GUICtrlSetOnEvent(-1, "_AcInputUDF_AutocompleteSelectedItem")
	$AcInputUDF_ACListGUIHandle_ESCDummy = GUICtrlCreateDummy()
	GUICtrlSetOnEvent(-1, "_AcInputUDF_HideAcList")
	Dim $AccelKeys[2][2] = [["{ENTER}", $AcInputUDF_ACListGUIHandle_EnterDummy], ["{ESC}", $AcInputUDF_ACListGUIHandle_ESCDummy]]
	GUISetAccelerators($AccelKeys, $AcInputUDF_ACListGUIHandle)

	;Set CallBackFunc
	$AcInputUDF_RegisteredCallBackFunc = $CallBackFunc

	;Register exit Func
	OnAutoItExitRegister("_AcInputUDF_ShutdownUDF")

	Return 1
EndFunc   ;==>_AcInputUDF_Initialize


; #FUNCTION# ====================================================================================================================
; Name...........: _AcInputUDF_RegisterControl
; Description ...: Register an GUICtrlCreateInput control for the AcInput UDF.
; Syntax.........: _AcInputUDF_RegisterControl($CntHandle = "",$OnClickFunc = "")
; Parameters ....: $CntHandle       - Handle to the GUICtrlCreateInput control
; 				   $OnClickFunc     - Defines a user-defined function to be called when an item in the listbox is clicked.
; Return values .: Success          - 1
;                  Failure          - -1
; Author ........: ISI360
; Modified.......:
; Remarks .......: The Subclass ID for _WinAPI_SetWindowSubclass starts cointing up from 3500
; Related .......:
; Link ..........:
; Example .......:
; ===============================================================================================================================
Func _AcInputUDF_RegisterControl($CntHandle = "", $OnClickFunc = "")
	If $CntHandle = "" Then Return -1
	If Not IsArray($AcInputUDF_RegisteredControlsArray) Then Return -1

	;Search, if the control is already added to our $AcInputUDF_RegisteredControlsArray
	Local $FoundIndex = _ArraySearch($AcInputUDF_RegisteredControlsArray, GUICtrlGetHandle($CntHandle))
	If Not @error Or $FoundIndex <> -1 Then Return -1

	;Add Handle to the $InputAutocomplete Array
	Local $ArrayIndex = _ArrayAdd($AcInputUDF_RegisteredControlsArray, GUICtrlGetHandle($CntHandle))
	If @error Then Return -1

	;Get owner window
	Local $OwnerWindow = _WinAPI_GetAncestor(GUICtrlGetHandle($CntHandle), $GA_PARENT)

	;Set Global Callback Subclass to the Control
	GUIRegisterMsg20($CntHandle, $WM_KEYDOWN, $AcInputUDF_RegisteredCallBackFunc)
	GUIRegisterMsg20($CntHandle, $WM_KEYUP, $AcInputUDF_RegisteredCallBackFunc)

	;Fill array
	$AcInputUDF_RegisteredControlsArray[$ArrayIndex][2] = $OwnerWindow
	$AcInputUDF_RegisteredControlsArray[$ArrayIndex][3] = $OnClickFunc

	;Set $AcInputUDF_RegisteredControlsAsString
	$AcInputUDF_RegisteredControlsAsString = _ArrayToString($AcInputUDF_RegisteredControlsArray, "|", -1, -1, "|", -1, 0)

	Return 1
EndFunc   ;==>_AcInputUDF_RegisterControl

; #FUNCTION# ====================================================================================================================
; Name...........: _AcInputUDF_UnRegisterControl
; Description ...: Unregisters a control from the AcInput UDF.
; Syntax.........: _AcInputUDF_UnRegisterControl($CntHandle = "")
; Parameters ....: $CntHandle       - Handle to the GUICtrlCreateInput control
; Return values .: Success          - 1
;                  Failure          - -1
; Author ........: ISI360
; Modified.......:
; Remarks .......:
; Related .......:
; Link ..........:
; Example .......:
; ===============================================================================================================================
Func _AcInputUDF_UnRegisterControl($CntHandle = "")
	If $CntHandle = "" Then Return -1
	If Not IsArray($AcInputUDF_RegisteredControlsArray) Then Return -1

	;Search, if the control is added to our $AcInputUDF_RegisteredControlsArray
	Local $FoundIndex = _ArraySearch($AcInputUDF_RegisteredControlsArray, GUICtrlGetHandle($CntHandle))
	If @error Or $FoundIndex = -1 Then Return -1

	;Delete the array
	_ArrayDelete($AcInputUDF_RegisteredControlsArray, $FoundIndex)

	;Unregister control from subclassing
	GUIUnRegisterMsg20($CntHandle, $WM_KEYDOWN, $AcInputUDF_RegisteredCallBackFunc)
	GUIUnRegisterMsg20($CntHandle, $WM_KEYUP, $AcInputUDF_RegisteredCallBackFunc)

	;Set $AcInputUDF_RegisteredControlsAsString
	$AcInputUDF_RegisteredControlsAsString = _ArrayToString($AcInputUDF_RegisteredControlsArray, "|", -1, -1, "|", -1, 0)

	Return 1
EndFunc   ;==>_AcInputUDF_UnRegisterControl

; #FUNCTION# ====================================================================================================================
; Name...........: _AcInputUDF_ControlSetAcList
; Description ...: Set/update the autocomplete list for an control.
; Syntax.........: _AcInputUDF_ControlSetAcList($CntHandle = "", $ListAsArray = "")
; Parameters ....: $CntHandle       - Handle to the GUICtrlCreateInput control
;                  $ListAsArray     - Keywoards as Array
; Return values .: Success          - 1
;                  Failure          - -1
; Author ........: ISI360
; Modified.......:
; Remarks .......:
; Related .......:
; Link ..........:
; Example .......:
; ===============================================================================================================================
Func _AcInputUDF_ControlSetAcList($CntHandle = "", $ListAsArray = "")
	If $CntHandle = "" Or $ListAsArray = "" Then Return -1
	If Not IsArray($ListAsArray) Then Return -1

	;Search, if the control is added to our $AcInputUDF_RegisteredControlsArray
	Local $FoundIndex = _ArraySearch($AcInputUDF_RegisteredControlsArray, GUICtrlGetHandle($CntHandle))
	If @error Or $FoundIndex = -1 Then Return -1

	;Set the autocomplete string
	$AcInputUDF_RegisteredControlsArray[$FoundIndex][1] = $ListAsArray
	Return 1
EndFunc   ;==>_AcInputUDF_ControlSetAcList

; #FUNCTION# ====================================================================================================================
; Name...........: _AcInputUDF_AutocompleteSelectedItem
; Description ...: Autocompletes the selected item and sends the text to the input control.
; Syntax.........: _AcInputUDF_AutocompleteSelectedItem()
; Parameters ....: None
; Return values .: Success          - 1
;                  Failure          - -1
; Author ........: ISI360
; Modified.......:
; Remarks .......:
; Related .......:
; Link ..........:
; Example .......:
; ===============================================================================================================================
Func _AcInputUDF_AutocompleteSelectedItem()
	If Not IsArray($AcInputUDF_RegisteredControlsArray) Then Return -1
	If Not BitAND(WinGetState($AcInputUDF_ACListGUIHandle, ""), 2) Then Return ;Gui is not visible
	Local $iCurrIndex = _GUICtrlListBox_GetCurSel($AcInputUDF_ACListControlHandle)
	If $iCurrIndex = -1 Or @error Then
		_AcInputUDF_HideAcList()
		Return -1
	EndIf
	GUICtrlSetData(_WinAPI_GetDlgCtrlID($AcInputUDF_LastActivatedControl), _GUICtrlListBox_GetText($AcInputUDF_ACListControlHandle, $iCurrIndex))
	_AcInputUDF_HideAcList()
	Return 1
EndFunc   ;==>_AcInputUDF_AutocompleteSelectedItem

; #FUNCTION# ====================================================================================================================
; Name...........: _AcInputUDF_SelectNextItemInList
; Description ...: Selects the next item in the autocomplete list. (If visible)
; Syntax.........: _AcInputUDF_SelectNextItemInList()
; Parameters ....: None
; Return values .: Success          - 1
;                  Failure          - -1
; Author ........: ISI360
; Modified.......:
; Remarks .......:
; Related .......:
; Link ..........:
; Example .......:
; ===============================================================================================================================
Func _AcInputUDF_SelectNextItemInList()
	If Not IsArray($AcInputUDF_RegisteredControlsArray) Then Return -1
	If Not BitAND(WinGetState($AcInputUDF_ACListGUIHandle, ""), 2) Then Return ;Gui is not visible

	Local $iCurrIndex = _GUICtrlListBox_GetCurSel($AcInputUDF_ACListControlHandle)
	$iCurrIndex = $iCurrIndex + 1
	If $iCurrIndex > _GUICtrlListBox_GetCount($AcInputUDF_ACListControlHandle) - 1 Then
		$iCurrIndex = 0
	EndIf
	_GUICtrlListBox_SetCurSel($AcInputUDF_ACListControlHandle, $iCurrIndex)

	;Autoscroll on long text
	_AcInputUDF_ListBoxUpdateView()

	Return 1
EndFunc   ;==>_AcInputUDF_SelectNextItemInList


; #FUNCTION# ====================================================================================================================
; Name...........: _AcInputUDF_SelectPrevItemInList
; Description ...: Selects the previous item in the autocomplete list. (If visible)
; Syntax.........: _AcInputUDF_SelectPrevItemInList()
; Parameters ....: None
; Return values .: Success          - 1
;                  Failure          - -1
; Author ........: ISI360
; Modified.......:
; Remarks .......:
; Related .......:
; Link ..........:
; Example .......:
; ===============================================================================================================================
Func _AcInputUDF_SelectPrevItemInList()
	If Not IsArray($AcInputUDF_RegisteredControlsArray) Then Return -1
	If Not BitAND(WinGetState($AcInputUDF_ACListGUIHandle, ""), 2) Then Return ;Gui is not visible

	Local $iCurrIndex = _GUICtrlListBox_GetCurSel($AcInputUDF_ACListControlHandle)
	$iCurrIndex = $iCurrIndex - 1
	If $iCurrIndex < 0 Then
		$iCurrIndex = _GUICtrlListBox_GetCount($AcInputUDF_ACListControlHandle) - 1
	EndIf
	_GUICtrlListBox_SetCurSel($AcInputUDF_ACListControlHandle, $iCurrIndex)

	;Autoscroll on long text
	_AcInputUDF_ListBoxUpdateView()

	Return 1
EndFunc   ;==>_AcInputUDF_SelectPrevItemInList

; #FUNCTION# ====================================================================================================================
; Name...........: _AcInputUDF_IsControlRegistered
; Description ...: Checks if an control is registered in the AcInput UDF or not.
; Syntax.........: _AcInputUDF_IsControlRegistered($CntHandle = "")
; Parameters ....: $CntHandle       - Handle to the GUICtrlCreateInput control to check
; Return values .: Success          - Handle to the control
;                  Failure          - Control is not registered
; Author ........: ISI360
; Modified.......:
; Remarks .......:
; Related .......:
; Link ..........:
; Example .......:
; ===============================================================================================================================
Func _AcInputUDF_IsControlRegistered($CntHandle = "")
	If $CntHandle = "" Then Return 0
	If Not IsArray($AcInputUDF_RegisteredControlsArray) Then Return 0

	;Search, if the control is added to our $AcInputUDF_RegisteredControlsAsString (we use $AcInputUDF_RegisteredControlsAsString for faster compare)
	If $AcInputUDF_RegisteredControlsAsString = "" Or Not StringInStr($AcInputUDF_RegisteredControlsAsString, String($CntHandle)) Then Return 0

	Return $CntHandle
EndFunc   ;==>_AcInputUDF_IsControlRegistered

; #FUNCTION# ====================================================================================================================
; Name...........: _AcInputUDF_HideAcList
; Description ...: Hides the autocomplete list (If displayed)
; Syntax.........: _AcInputUDF_HideAcList()
; Parameters ....: None
; Return values .: Success          - 1
;                  Failure          - -1
; Author ........: ISI360
; Modified.......:
; Remarks .......:
; Related .......:
; Link ..........:
; Example .......:
; ===============================================================================================================================
Func _AcInputUDF_HideAcList()
	If Not IsArray($AcInputUDF_RegisteredControlsArray) Then Return -1
	If Not BitAND(WinGetState($AcInputUDF_ACListGUIHandle, ""), 2) Then Return ;Gui is already hidden
	$AcInputUDF_ACListLastInsertedText = "" ;Reset last input
	HotKeySet("{ESC}") ;Remove global ESC hotkey
	GUISetState(@SW_HIDE, $AcInputUDF_ACListGUIHandle)
EndFunc   ;==>_AcInputUDF_HideAcList

; #FUNCTION# ====================================================================================================================
; Name...........: _AcInputUDF_ListboxSetFont
; Description ...: Sets the Font and its sizes for the listbox control
; Syntax.........: _AcInputUDF_ListboxSetFont($FontName = "Arial", $FontSize = 8.5, $FontWeight = 0, $FontAttribute = 0)
; Parameters ....: $FontName       - Font to set
;                  $FontSize       - Font size to set
;                  $FontWeight     - Font weight to set
;                  $FontAttribute  - Font Attributes to set
;                  See "GUICtrlSetFont" for more infos about the parameters.
; Return values .: See GUICtrlSetFont
; Author ........: ISI360
; Modified.......:
; Remarks .......:
; Related .......:
; Link ..........:
; Example .......:
; ===============================================================================================================================
Func _AcInputUDF_ListboxSetFont($FontName = "Arial", $FontSize = 8.5, $FontWeight = 0, $FontAttribute = 0)
	If Not IsArray($AcInputUDF_RegisteredControlsArray) Then Return -1
	Return GUICtrlSetFont($AcInputUDF_ACListControlHandle, $FontSize, $FontWeight, $FontAttribute, $FontName)
EndFunc   ;==>_AcInputUDF_ListboxSetFont

; #FUNCTION# ====================================================================================================================
; Name...........: _AcInputUDF_ListboxSetColour
; Description ...: Sets the Colors of the listbox control
; Syntax.........: _AcInputUDF_ListboxSetFont($ForegroundColor = 0x000000, $BackgroundColor = 0xFFFFFF)
; Parameters ....: $ForegroundColor       - The foreground color
;                  $BackgroundColor       - The background color
;                  See "GUICtrlSetColor" and "GUICtrlSetBkColor" for more infos about the parameters.
; Return values .: Always return 1
; Author ........: ISI360
; Modified.......:
; Remarks .......:
; Related .......:
; Link ..........:
; Example .......:
; ===============================================================================================================================
Func _AcInputUDF_ListboxSetColour($ForegroundColor = 0x000000, $BackgroundColor = 0xFFFFFF)
	If Not IsArray($AcInputUDF_RegisteredControlsArray) Then Return -1
	GUICtrlSetColor($AcInputUDF_ACListControlHandle, $ForegroundColor)
	GUICtrlSetBkColor($AcInputUDF_ACListControlHandle, $BackgroundColor)
	Return 1
EndFunc   ;==>_AcInputUDF_ListboxSetColour

; #FUNCTION# ====================================================================================================================
; Name...........: _AcInputUDF_ShowAcList
; Description ...: Display the Autocomplete list to an control
; Syntax.........: _AcInputUDF_ShowAcList($InputControl="")
; Parameters ....: $CntHandle       - Handle to the control
; Return values .: Success          - 1
;                  Failure          - -1
; Author ........: ISI360
; Modified.......:
; Remarks .......: Internal use only!
; Related .......:
; Link ..........:
; Example .......:
; ===============================================================================================================================
Func _AcInputUDF_ShowAcList($InputControl = "")
	If Not IsArray($AcInputUDF_RegisteredControlsArray) Then Return -1

	If $InputControl <> $AcInputUDF_ACListLastInsertedText Then $AcInputUDF_ACListControlPos = _ControlGetPos("", "", $InputControl)
	If Not IsArray($AcInputUDF_ACListControlPos) Then Return -1

	$AcInputUDF_LastActivatedControl = GUICtrlGetHandle($InputControl)

	If BitAND(WinGetState($AcInputUDF_ACListGUIHandle, ""), 2) Then Return _AcInputUDF_ReposGui() ;Gui is already visible

	;Pre-Resize the Autocomplete List
	Local $OwnerWindow = _WinAPI_GetAncestor(GUICtrlGetHandle($InputControl), $GA_PARENT)

	;Set the owner of the autocomplete list
	_WinAPI_SetWindowLong($AcInputUDF_ACListGUIHandle, $GWL_HWNDPARENT, $OwnerWindow)

	;Set a global ESC hotkey
	HotKeySet("{ESC}", "_AcInputUDF_HideAcList")

	;Set GUI position and show the GUI
	_AcInputUDF_ReposGui()
	GUISetState(@SW_SHOWNOACTIVATE, $AcInputUDF_ACListGUIHandle)
	WinSetOnTop($AcInputUDF_ACListGUIHandle, "", 1)


	AdlibRegister("_AcInputUDF_AcCheckFocusAdlib", 150)
	Return 1
EndFunc   ;==>_AcInputUDF_ShowAcList

; #FUNCTION# ====================================================================================================================
; Name...........: _AcInputUDF_CheckInput
; Description ...: Checks the input and search for a match in the autocomplete list
; Syntax.........: _AcInputUDF_CheckInput($InputControl="")
; Parameters ....: $InputControl       - Handle to the control
; Return values .: Success          - 1
;                  Failure          - -1
; Author ........: ISI360
; Modified.......:
; Remarks .......: Internal use only!
; Related .......:
; Link ..........:
; Example .......:
; ===============================================================================================================================
Func _AcInputUDF_CheckInput($InputControl = "")
	If Not IsArray($AcInputUDF_RegisteredControlsArray) Then Return -1


	If GUICtrlRead($InputControl) <> $AcInputUDF_ACListLastInsertedText Or $AcInputUDF_LastActivatedControl <> GUICtrlGetHandle($InputControl) Then ; Input content or pointer are changed.
		$AcInputUDF_ACListLastInsertedText = GUICtrlRead($InputControl)

		Local $sPartialData = ""
		Local $FoundIndex = _ArraySearch($AcInputUDF_RegisteredControlsArray, GUICtrlGetHandle($InputControl))
		If @error Or $FoundIndex = -1 Then Return -1
		GUICtrlSetOnEvent($AcInputUDF_ACListControlHandle, $AcInputUDF_RegisteredControlsArray[$FoundIndex][3])
		Local $AcData = $AcInputUDF_RegisteredControlsArray[$FoundIndex][1] ;Get Array
		If Not IsArray($AcData) Then Return -1
		_GUICtrlListBox_BeginUpdate($AcInputUDF_ACListControlHandle)
		GUICtrlSetData($AcInputUDF_ACListControlHandle, "") ;Clear Data

		For $A = 0 To UBound($AcData) - 1
			If StringInStr($AcData[$A], $AcInputUDF_ACListLastInsertedText) Then
				$sPartialData &= $AcData[$A] & "|"
			EndIf
		Next
		GUICtrlSetData($AcInputUDF_ACListControlHandle, $sPartialData) ;Set List Data
		_GUICtrlListBox_UpdateHScroll($AcInputUDF_ACListControlHandle)
		_GUICtrlListBox_EndUpdate($AcInputUDF_ACListControlHandle)

		Local $ItemCount = _GUICtrlListBox_GetCount($AcInputUDF_ACListControlHandle) ;Get item Count
		If $ItemCount > 0 Then
			If $ItemCount = 1 And _GUICtrlListBox_GetText($AcInputUDF_ACListControlHandle, 0) = $AcInputUDF_ACListLastInsertedText Then
				_AcInputUDF_HideAcList()
			Else
				_AcInputUDF_ShowAcList($InputControl)
			EndIf
		Else
			_AcInputUDF_HideAcList()
		EndIf
	EndIf

	Return 1
EndFunc   ;==>_AcInputUDF_CheckInput

; #FUNCTION# ====================================================================================================================
; Name...........: _AcInputUDF_ReposGui
; Description ...: Repos the Autocomplete Gui
; Syntax.........: _AcInputUDF_ReposGui()
; Parameters ....: None
; Return values .: Success          - 1
;                  Failure          - -1
; Author ........: ISI360
; Modified.......:
; Remarks .......: Internal use only!
; Related .......:
; Link ..........:
; Example .......:
; ===============================================================================================================================
Func _AcInputUDF_ReposGui()
	Local $OwnerWindow = _WinAPI_GetAncestor($AcInputUDF_LastActivatedControl, $GA_PARENT)
	Local $AcListCount = _GUICtrlListBox_GetCount($AcInputUDF_ACListControlHandle)
	Local $Itemheight = _GUICtrlListBox_GetItemHeight($AcInputUDF_ACListControlHandle)

	Local $tPoint = DllStructCreate("int X;int Y")
	DllStructSetData($tPoint, "X", $AcInputUDF_ACListControlPos[0])
	DllStructSetData($tPoint, "Y", $AcInputUDF_ACListControlPos[1])
	_WinAPI_ClientToScreen($OwnerWindow, $tPoint)


	If _GUICtrlListBox_GetHorizontalExtent($AcInputUDF_ACListControlHandle) > $AcInputUDF_ACListControlPos[2] Then
		Local $NewWindowHeight = $Itemheight * ($AcListCount + 3)
	Else
		Local $NewWindowHeight = $Itemheight * ($AcListCount + 1)
	EndIf

	If $NewWindowHeight > $AcInputUDF_AutoListMaxHeight Then $NewWindowHeight = $AcInputUDF_AutoListMaxHeight

	Local $Currentpos = _WinGetPos($AcInputUDF_ACListGUIHandle)
	If Not IsArray($Currentpos) Then Return
	If $Currentpos[0] <> DllStructGetData($tPoint, "X") Or $Currentpos[1] <> DllStructGetData($tPoint, "Y") + $AcInputUDF_ACListControlPos[3] + 2 Or $Currentpos[2] <> $AcInputUDF_ACListControlPos[2] Or $Currentpos[3] <> $NewWindowHeight Then ;Move only if needed
		WinMove($AcInputUDF_ACListGUIHandle, "", DllStructGetData($tPoint, "X"), DllStructGetData($tPoint, "Y") + $AcInputUDF_ACListControlPos[3] + 2, $AcInputUDF_ACListControlPos[2], $NewWindowHeight)
		GUICtrlSetPos($AcInputUDF_ACListControlHandle, 0, 0, $AcInputUDF_ACListControlPos[2], $NewWindowHeight)
	EndIf
EndFunc   ;==>_AcInputUDF_ReposGui

; #FUNCTION# ====================================================================================================================
; Name...........: _AcInputUDF_AcCheckFocusAdlib
; Description ...: Checks if the last triggered autocomplete input loses its focus.
; Syntax.........: _AcInputUDF_AcCheckFocusAdlib()
; Parameters ....: None
; Return values .: None
; Author ........: ISI360
; Modified.......:
; Remarks .......: Internal use only!
; Related .......:
; Link ..........:
; Example .......:
; ===============================================================================================================================
Func _AcInputUDF_AcCheckFocusAdlib()
	Local $CurrentControl = _WinAPI_GetFocus()
	If $AcInputUDF_LastActivatedControl <> $CurrentControl And $CurrentControl <> GUICtrlGetHandle($AcInputUDF_ACListControlHandle) And WinGetHandle("[active]") <> $AcInputUDF_ACListGUIHandle Then
		_AcInputUDF_HideAcList()
		AdlibUnRegister("_AcInputUDF_AcCheckFocusAdlib")
	EndIf
	_AcInputUDF_ReposGui()
EndFunc   ;==>_AcInputUDF_AcCheckFocusAdlib

; #FUNCTION# ====================================================================================================================
; Name...........: _AcInputUDF_ShutdownUDF
; Description ...: Exits the AcInput UDF. It deletes the Autocomplete GUI and closes the CallBack func.
; Syntax.........: _AcInputUDF_ShutdownUDF()
; Parameters ....: None
; Return values .: Success          - 1
;                  Failure          - -1
; Author ........: ISI360
; Modified.......:
; Remarks .......: Internal use only!
; Related .......:
; Link ..........:
; Example .......:
; ===============================================================================================================================
Func _AcInputUDF_ShutdownUDF()
	If $AcInputUDF_RegisteredCallBackFunc = "" Then Return -1 ;Not initialized

	AdlibUnRegister("_AcInputUDF_AcCheckFocusAdlib")
	_AcInputUDF_HideAcList()

	;Remove subclassing from controls
	For $cnt = 0 To UBound($AcInputUDF_RegisteredControlsArray) - 1
		GUIUnRegisterMsg20($AcInputUDF_RegisteredControlsArray[$cnt][0], $WM_KEYDOWN, $AcInputUDF_RegisteredCallBackFunc)
		GUIUnRegisterMsg20($AcInputUDF_RegisteredControlsArray[$cnt][0], $WM_KEYUP, $AcInputUDF_RegisteredCallBackFunc)
	Next

	;Reset some stuff
	Local $AcInputUDF_RegisteredControlsArray_New[0][4]
	$AcInputUDF_RegisteredControlsArray = $AcInputUDF_RegisteredControlsArray_New
	$AcInputUDF_RegisteredControlsAsString = ""

	;Delete the autocomeplete GUI
	GUIDelete($AcInputUDF_ACListGUIHandle)
	$AcInputUDF_ACListGUIHandle = ""

	;Clear CallbackFunc
	$AcInputUDF_RegisteredCallBackFunc = ""

	Return 1
EndFunc   ;==>_AcInputUDF_ShutdownUDF

; #FUNCTION# ====================================================================================================================
; Name...........: _AcInputUDF_ListBoxUpdateView
; Description ...: Scrolls the view of the selected item in the listbox. If the textlenght is over the box width -> Scroll to lineend, if not scroll to line start
; Syntax.........: _AcInputUDF_ListBoxUpdateView()
; Parameters ....: None
; Return values .: Success          - 1
;                  Failure          - -1
; Author ........: ISI360
; Modified.......:
; Remarks .......: Internal use only!
; Related .......:
; Link ..........:
; Example .......:
; ===============================================================================================================================
Func _AcInputUDF_ListBoxUpdateView()
	If Not IsArray($AcInputUDF_ACListControlPos) Then Return -1

	Local $Index = _GUICtrlListBox_GetCurSel($AcInputUDF_ACListControlHandle)
	Local $hFont = _SendMessage($AcInputUDF_ACListControlHandle, $__LISTBOXCONSTANT_WM_GETFONT)
	Local $hDC = _WinAPI_GetDC($AcInputUDF_ACListControlHandle)
	_WinAPI_SelectObject($hDC, $hFont)

	$sText = _GUICtrlListBox_GetText($AcInputUDF_ACListControlHandle, $Index)
	$tSize = _WinAPI_GetTextExtentPoint32($hDC, $sText & "W")


	_WinAPI_SelectObject($hDC, $hFont)
	_WinAPI_ReleaseDC($AcInputUDF_ACListControlHandle, $hDC)


	;Autoscroll on long text
	If DllStructGetData($tSize, "X") > $AcInputUDF_ACListControlPos[2] Then
		GUICtrlSendMsg($AcInputUDF_ACListControlHandle, $WM_HSCROLL, $SB_PAGEDOWN, 0)
		GUICtrlSendMsg($AcInputUDF_ACListControlHandle, $WM_HSCROLL, $SB_PAGEDOWN, 0)
	Else
		GUICtrlSendMsg($AcInputUDF_ACListControlHandle, $WM_HSCROLL, $SB_PAGEUP, 0)
		GUICtrlSendMsg($AcInputUDF_ACListControlHandle, $WM_HSCROLL, $SB_PAGEUP, 0)
	EndIf

EndFunc   ;==>_AcInputUDF_ListBoxUpdateView

